using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.Common.Extensions;
using System;
using System.Collections;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;

namespace gov.va.med.vbecs.BOL
{
    /// <summary>
    /// PendingBloodUnitTest, used to get pending blood units for user review
    /// </summary>
    public class PendingBloodUnitTest
    {
        /// <summary>
        /// Call data access layer to return data table of blood unit test results for given blood unit and test type id.
        /// </summary>
        /// <param name="divisionCode"></param>
        /// <param name="pendingTestStatus"></param>
        /// <returns></returns>
        public static IList<IPendingBloodUnitTestModel> GetPendingBloodUnitTests(string divisionCode, PendingTestStatus pendingTestStatus)
        {
            var pendingTestList = new List<IPendingBloodUnitTestModel>();

            // Get PendingBloodUnitTestResults from the database
            var dsAllPendingBloodUnitTests = DAL.PendingBloodUnitTest.GetPendingBloodUnitTestResults(divisionCode, pendingTestStatus);
            // Convert DataTable to list of PendingBloodUnitTestDao
            var pendingBloodUnitTestDaoList = dsAllPendingBloodUnitTests.Tables[0].ToList<PendingBloodUnitTestDao>();
            // Convert DataTable to list of PendingBloodUnitDao
            var pendingBloodUnitDaoList = dsAllPendingBloodUnitTests.Tables[1].ToList<PendingBloodUnitDao>();
            // Group test records by the ResultSetGuid
            var resultSetList = pendingBloodUnitTestDaoList.GroupBy(g => g.UnitResultSetGuid);

            // Loop through each set of results and build the PendingBloolUnitAboRhModel and PendingBloodUnitAntigenTypingModel
            foreach (var resultSet in resultSetList)
            {
                var firstTestDao = resultSet.First();
                var pendingBloodUnitDao = pendingBloodUnitDaoList.Where(x => x.BloodUnitGuid == firstTestDao.BloodUnitGuid).FirstOrDefault();
                var resultOrderableTest = (OrderableTest)firstTestDao.OrderableTestId;

                if (PendingTestHelper.IsAntigenTypingOrderableTest(resultOrderableTest))
                {
                    pendingTestList.Add(BuildPendingBloolUnitAntigenTypingModel(resultSet.ToList(), pendingBloodUnitDao));
                }
                else if (resultOrderableTest == OrderableTest.ABORh ||
                         resultOrderableTest == OrderableTest.RepeatABORh)
                {
                    pendingTestList.Add(BuildPendingBloodUnitAboRhModel(resultSet.ToList(), pendingBloodUnitDao));
                }
            }

            return pendingTestList;
        }

        /// <summary>
        /// BuildPendingBloolUnitAboRhModel used to build a Blood Unit AboRh record
        /// </summary>
        /// <param name="pendingBloodUnitTestDaoList"></param>
        /// <param name="pendingBloodUnitDao"></param>
        /// <returns></returns>
        private static PendingBloodUnitAboRhModel BuildPendingBloodUnitAboRhModel(IList<PendingBloodUnitTestDao> pendingBloodUnitTestDaoList, PendingBloodUnitDao pendingBloodUnitDao)
        {
            var aboRhModel = new PendingBloodUnitAboRhModel();

            var firstDao = pendingBloodUnitTestDaoList.First();

            // Assign Blood Unit Base Data
            PendingTestHelper.AssignPropertiesFromDao(aboRhModel, pendingBloodUnitTestDaoList);

            // Assign Abo/Rh specific properties            
            aboRhModel.BloodUnitId = firstDao.BloodUnitId;
            aboRhModel.OrderedTestGuid = firstDao.OrderedTestGuid;
            aboRhModel.OrderedComponentGuid = firstDao.OrderedComponentGuid;
            aboRhModel.ResultSetGuid = firstDao.UnitResultSetGuid;
            aboRhModel.BloodUnitModel = PendingTestHelper.GetBloodUnitModel(pendingBloodUnitDao);
            aboRhModel.TestValidation = aboRhModel.Validate(aboRhModel.BloodUnitModel);

            return aboRhModel;
        }

        /// <summary>
        /// BuildPendingBloolUnitAboRhModel used to build a Blood Unit AboRh record
        /// </summary>
        /// <param name="pendingBloodUnitTestDaoList"></param>
        /// <param name="pendingBloodUnitDao"></param>
        /// <returns></returns>
        private static PendingBloodUnitAntigenTypingModel BuildPendingBloolUnitAntigenTypingModel(IList<PendingBloodUnitTestDao> pendingBloodUnitTestDaoList, PendingBloodUnitDao pendingBloodUnitDao)
        {
            var antigenTypingModel = new PendingBloodUnitAntigenTypingModel();

            var firstDao = pendingBloodUnitTestDaoList.First();

            // Assign Blood Unit Base Data
            PendingTestHelper.AssignPropertiesFromDao(antigenTypingModel, pendingBloodUnitTestDaoList);

            // Assign Antigen Typign specific propertiest
            antigenTypingModel.BloodUnitId = firstDao.BloodUnitId;
            antigenTypingModel.OrderedTestGuid = firstDao.OrderedTestGuid;
            antigenTypingModel.OrderedComponentGuid = firstDao.OrderedComponentGuid;
            antigenTypingModel.ResultSetGuid = firstDao.UnitResultSetGuid;
            antigenTypingModel.BloodUnitModel = PendingTestHelper.GetBloodUnitModel(pendingBloodUnitDao);
            antigenTypingModel.TestValidation = antigenTypingModel.Validate(antigenTypingModel.BloodUnitModel);

            return antigenTypingModel;
        }

        /// <summary>
        /// Determine if test results exist for a given divisiona and PendingTestStatus
        /// </summary>
        /// <param name="divisionCode"></param>
        /// <param name="pendingTestStatus"></param>
        /// <returns>true if tests exist, otherwise false</returns>
        public static bool DoTestsExist(string divisionCode, PendingTestStatus pendingTestStatus)
        {
            return DAL.PendingBloodUnitTest.GetCount(divisionCode, pendingTestStatus) > 0;
        }

        /// <summary>
        /// RejectTests
        /// </summary>
        /// <param name="pendingTestList"></param>
        public static bool RejectTests(IList<IPendingBloodUnitTestModel> pendingTestList)
        {
            if (pendingTestList == null || !pendingTestList.Any()) return false;

            // Buid DataTable to update/reject Pending Blood Unit Tests
            DataTable dtPendingTestReject = DAL.PendingBloodUnitTest.GetEmptyPendingBloodUnitTestTableSchemaForUpdate();
            foreach (var pendingTest in pendingTestList)
            {
                if (string.IsNullOrWhiteSpace(pendingTest.ReviewComment))
                {
                    throw new Exception(StrRes.ValidMsg.UC115.ReviewCommentRequired().ResString);
                }

                foreach (var testResult in pendingTest.TestResultList)
                {
                    DataRow dr = dtPendingTestReject.NewRow();
                    dr[TABLE.PendingBloodUnitTest.PendingBloodUnitTestId] = testResult.PendingTestId;
                    dr[TABLE.PendingBloodUnitTest.BloodUnitTestGuid] = DBNull.Value;
                    dr[TABLE.PendingBloodUnitTest.PendingTestStatusId] = (byte)PendingTestStatus.Rejected;
                    dr[TABLE.PendingBloodUnitTest.RejectionComment] = pendingTest.ReviewComment;
                    dr[TABLE.PendingBloodUnitTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
                    dr[TABLE.PendingBloodUnitTest.RowVersion] = testResult.RowVersion;
                    dr[TABLE.PendingBloodUnitTest.LastUpdateFunctionId] = (int)UpdateFunction.UC115BloodUnitTabView;
                    dtPendingTestReject.Rows.Add(dr);
                }
            }

            // Update Pending Tests Status
            return DAL.PendingBloodUnitTest.Update(dtPendingTestReject);
        }

        /// <summary>
        /// Accept Tests
        /// BR_2.12 
        /// </summary>
        /// <param name="divisionCode"></param>
        /// <param name="pendingTestList"></param>
        /// <param name="isWorkLoadDefined"></param>
        public static bool AcceptTests(string divisionCode, IList<IPendingBloodUnitTestModel> pendingTestList, ref bool isWorkLoadDefined)
        {
            if (pendingTestList == null || !pendingTestList.Any()) return false;

            ArrayList allTables = new ArrayList();
            ArrayList allSprocs = new ArrayList();

            // Add to BloodUnitTests and Update PendingBloodUnitTests
            DataTable dtBloodUnitTests = null;
            DataTable dtPendingBloodUnitTests = null;
            DataTable dtBloodUnitStatus = null;
            BuildBloodUnitTestTablesForAccepting(divisionCode, pendingTestList, ref dtBloodUnitTests, ref dtBloodUnitStatus, ref dtPendingBloodUnitTests);

            allTables.Add(dtBloodUnitTests);
            allSprocs.Add(STOREDPROC.InsertBloodUnitTest.StoredProcName);

            allTables.Add(dtPendingBloodUnitTests);
            allSprocs.Add(STOREDPROC.UpdatePendingBloodUnitTest.StoredProcName);

            // Append Blood Unit Status
            if (dtBloodUnitStatus != null &&
                dtBloodUnitStatus.Rows.Count > 0)
            {
                allTables.Add(dtBloodUnitStatus);
                allSprocs.Add(STOREDPROC.UpdateBloodUnitStatus.StoredProcName);
            }

            // Generate Exception Report for each ExceptionType
            // Append on Exception Reports
            var exceptionReportList = ExceptionReportCreator.CreateExceptionReports(pendingTestList);
            if (exceptionReportList != null &&
                exceptionReportList.Any())
            {
                foreach (var exception in exceptionReportList)
                {
                    DataTable dtException = exception.ExceptionData.Table;
                    dtException.Rows.Add(exception.ExceptionData.ItemArray);
                    allTables.Add(dtException);
                    allSprocs.Add(ExceptionReport.GetStoredProcName(exception.ExceptionType));
                }
            }

            // Append WorkLoads
            DataTable dtWorkLoad = null;
            PendingTestHelper.BuildWorkLoadTableForAccepting(pendingTestList, UpdateFunction.UC115BloodUnitTabView, ref dtWorkLoad, ref isWorkLoadDefined);

            if (dtWorkLoad != null &&
                dtWorkLoad.Rows.Count > 0)
            {
                allTables.Add(dtWorkLoad);
                allSprocs.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
            }

            // Execute Transactions
            int retVal = new Common.StoredProcedure().TransactionalGetValue(allSprocs, allTables);

            return (retVal == 0);
        }

        /// <summary>
        /// Build data in DataTables for BloodUnitTest, BloodUnitStatus, and PendingloodUnitTest
        /// </summary>
        /// <param name="divisionCode"></param>
        /// <param name="pendingTestList"></param>
        /// <param name="dtBloodUnitTests"></param>
        /// <param name="dtBloodUnitStatus"></param>
        /// <param name="dtPendingBloodUnitTests"></param>
        private static void BuildBloodUnitTestTablesForAccepting(string divisionCode, IEnumerable<IPendingBloodUnitTestModel> pendingTestList, ref DataTable dtBloodUnitTests, ref DataTable dtBloodUnitStatus, ref DataTable dtPendingBloodUnitTests)
        {
            dtBloodUnitTests = DAL.BloodUnitTest.GetEmptyBloodUnitTestTableSchema(false);
            dtPendingBloodUnitTests = DAL.PendingBloodUnitTest.GetEmptyPendingBloodUnitTestTableSchemaForUpdate();
            dtBloodUnitStatus = DAL.BloodUnitStatus.GetEmptyBloodUnitStatusTableSchema(true);

            // Get canned comment for quarantining unit with inconsistent ABO/Rh tests
            DataTable quarantineCommentTable = DAL.CannedComment.GetCannedComments(divisionCode, "QD");
            Guid quarantineCommentGuid = (Guid)quarantineCommentTable.Rows[0][TABLE.CannedComment.CannedCommentGuid];

            foreach (var pendingTest in pendingTestList)
            {
                if (pendingTest.BloodUnitModel == null)
                {
                    throw new Exception("ProductCode must be selected before saving");
                }
                if (pendingTest.TestValidation.TestValidationStatus != TestValidationStatus.Valid &&
                    pendingTest.TestValidation.TestValidationStatus != TestValidationStatus.Warning)
                {
                    throw new Exception("Cannot accept a test that is not in a Valid or Warning validation status.");
                }

                foreach (var testResult in pendingTest.TestResultList)
                {
                    // Skip check cells for non-weak D typing
                    if (!pendingTest.OrderableTest.IsWeakD &&
                        testResult.TestType == TestType.Anti_Dc_AHG)
                    {
                        continue;
                    }

                    // Add to the BloodUnitTestTable
                    var drBUT = dtBloodUnitTests.NewRow();
                    Guid bloodUnitTestGuid = Guid.NewGuid();
                    drBUT[TABLE.BloodUnitTest.BloodUnitTestGuid] = bloodUnitTestGuid;
                    drBUT[TABLE.BloodUnitTest.BloodUnitGuid] = pendingTest.BloodUnitModel.BloodUnitGuid;
                    drBUT[TABLE.BloodUnitTest.BloodTestTypeId] = testResult.TestType;
                    drBUT[TABLE.BloodUnitTest.TestResultId] = testResult.TestResultId;
                    drBUT[TABLE.BloodUnitTest.TestDate] = pendingTest.TestedDateTime;
                    drBUT[TABLE.BloodUnitTest.TestTechId] = pendingTest.InstrumentUserId;
                    drBUT[TABLE.BloodUnitTest.EntryMethodCode] = Convert.ToString(Common.TestEntryMethod.Manual);
                    if (testResult.IsInterp)
                    {
                        drBUT[TABLE.BloodUnitTest.TestComments] = pendingTest.ReviewComment;
                    }
                    else
                    {
                        drBUT[TABLE.BloodUnitTest.TestComments] = string.Empty;
                    }
                    drBUT[TABLE.BloodUnitTest.DivisionCode] = pendingTest.DivisionCode;
                    drBUT[TABLE.BloodUnitTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
                    drBUT[TABLE.BloodUnitTest.LastUpdateFunctionId] = (int)UpdateFunction.UC115BloodUnitTabView;
                    drBUT[TABLE.BloodUnitTest.ConfirmationWorklistUnitGuid] = DBNull.Value;
                    drBUT[TABLE.BloodUnitTest.EntryTechId] = Common.LogonUser.LogonUserName;
                    drBUT[TABLE.BloodUnitTest.TestingMethodCode] = Common.TestingMethod.AutomatedInstrument;
                    drBUT[TABLE.BloodUnitTest.RackGuid] = DBNull.Value;
                    drBUT[TABLE.BloodUnitTest.AutoInstrumentName] = pendingTest.InstrumentName;
                    drBUT[TABLE.BloodUnitTest.RecordStatusCode] = RecordStatusCode.Active;
                    dtBloodUnitTests.Rows.Add(drBUT);

                    // Link to the PendingBloodUnitTest Table and update it's status
                    var drPendingUnit = dtPendingBloodUnitTests.NewRow();
                    drPendingUnit[TABLE.PendingBloodUnitTest.PendingBloodUnitTestId] = testResult.PendingTestId;
                    drPendingUnit[TABLE.PendingBloodUnitTest.BloodUnitTestGuid] = bloodUnitTestGuid;
                    drPendingUnit[TABLE.PendingBloodUnitTest.PendingTestStatusId] = (byte)PendingTestStatus.Accepted;
                    drPendingUnit[TABLE.PendingBloodUnitTest.RejectionComment] = DBNull.Value;
                    drPendingUnit[TABLE.PendingBloodUnitTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
                    drPendingUnit[TABLE.PendingBloodUnitTest.LastUpdateFunctionId] = (int)UpdateFunction.UC115BloodUnitTabView;
                    drPendingUnit[TABLE.PendingBloodUnitTest.RowVersion] = testResult.RowVersion;
                    dtPendingBloodUnitTests.Rows.Add(drPendingUnit);
                }

                // Update Blood Unit Status
                if (pendingTest.TestValidation.Quarantine)
                {
                    AppendQuarantineBloodUnitStatusRow(pendingTest.BloodUnitModel.BloodUnitGuid, pendingTest.BloodUnitModel.BloodUnitStatusRowVersion, quarantineCommentGuid, ref dtBloodUnitStatus);
                }
            }

            if (dtBloodUnitStatus != null)
            {
                Common.Utility.AppendLastUpdateInformation(dtBloodUnitStatus, UpdateFunction.UC115BloodUnitTabView);
            }
        }

        private static void AppendQuarantineBloodUnitStatusRow(Guid bloodUnitGuid, byte[] bloodUnitStatusRowVersion, Guid quarantineCommentGuid, ref DataTable dtBloodUnitStatus)
        {
            DataTable dtExistingBUSData = DAL.BloodUnitStatus.GetBloodUnitStatusByGuid(bloodUnitGuid);

            if (dtExistingBUSData.Rows.Count == 0)
            {
                throw new Exception("Existing Blood Unit Status not found");
            }
            else if (dtExistingBUSData.Rows.Count > 1)
            {
                throw new Exception("Multiple Blood Unit Status records found");
            }

            // Only quarantine if it's not already
            if ((bool)dtExistingBUSData.Rows[0][TABLE.BloodUnitStatus.QuarantineIndicator] == false)
            {
                DataRow drBUS = dtBloodUnitStatus.NewRow();
                foreach (System.Data.DataColumn dc in dtBloodUnitStatus.Columns)
                {
                    if (dtExistingBUSData.Columns.Contains(dc.ColumnName))
                    {
                        if (dc.ColumnName == TABLE.BloodUnitStatus.RowVersion)
                        {
                            //Use the original rowversion, so we're not cheating and potentially updating modified data since our tests began
                            drBUS[dc.ColumnName] = bloodUnitStatusRowVersion;
                        }
                        else
                        {
                            drBUS[dc.ColumnName] = dtExistingBUSData.Rows[0][dc.ColumnName];
                        }
                    }
                }

                //Now, change the blood unit status as appropriate                
                drBUS[TABLE.BloodUnitStatus.TestingLimitIndicator] = true;
                drBUS[TABLE.BloodUnitStatus.QuarantineIndicator] = 1;
                drBUS[TABLE.BloodUnitStatus.QuarantineCannedCommentGuid] = quarantineCommentGuid;

                dtBloodUnitStatus.Rows.Add(drBUS);
            }
        }

        /// <summary>
        /// A Pending Blood Unit could match multiple product codes, this method will return
        /// the valid matches. Returned blood units are based on PT_1.06 using the ProductType table's 
        /// AboRhConfirmationRequired flag == 1
        /// </summary>
        /// <param name="divisionCode"></param>
        /// <param name="bloodUnitId"></param>
        /// <param name="specimenUid"></param>
        /// <returns>true if tests exist, otherwise false</returns>
        public static IEnumerable<BloodUnitModel> GetValidBloodUnits(string divisionCode, string bloodUnitId, string specimenUid)
        {
            var prouctCodeList = new List<BloodUnitModel>();
            var pendingBloodUnitTestDaoList = new List<BloodUnitDao>();

            if (string.IsNullOrWhiteSpace(specimenUid))
            {
                var dtBloodUnits = DAL.PendingBloodUnitTest.GetBloodUnitsByUnitId(divisionCode, bloodUnitId);
                pendingBloodUnitTestDaoList = dtBloodUnits.ToList<BloodUnitDao>().OrderBy(x => x.ProductTypeSortColumn).ToList();
            }
            else
            {
                // if we have specimenUid it must be a XM, which changes how we need to search
                // Defect 282082
                var dtBloodUnits = DAL.PendingBloodUnitTest.GetXMBloodUnitsBySpecimenUIDAndUnitId(divisionCode, bloodUnitId, specimenUid);
                pendingBloodUnitTestDaoList = dtBloodUnits.ToList<BloodUnitDao>().OrderBy(x => x.ProductTypeSortColumn).ToList();
            }

            foreach (var bloodUnit in pendingBloodUnitTestDaoList)
            {
                var bloodUnitModel = PendingTestHelper.GetBloodUnitModel(bloodUnit);
                prouctCodeList.Add(bloodUnitModel);
            }

            return prouctCodeList;
        }
    }
}



